//*************************************************************************************************
//
//	Description:
//		MotionBlur2.fx
//
//	<P> Copyright (c) 2008 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Alastair Murray
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		AMurray		    01/03/2008  0.1           Created
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"

// default to medium quality
#if !defined(HIGH_QUALITY) && !defined(MEDIUM_QUALITY) && !defined(LOW_QUALITY)
#define	MEDIUM_QUALITY
#endif

// Works on all platforms
//#define	ANISO_TRICK

#if defined(HIGH_QUALITY)
	#define	SAMPLES		12
	#define	DIVVEL		12
	#define	MAX_ANISO	16
#elif defined(MEDIUM_QUALITY)
	#define	SAMPLES		8
	#define	DIVVEL		8
	#define	MAX_ANISO	16
#elif defined(LOW_QUALITY)
	#define	SAMPLES		4
	#define	DIVVEL		4
	#define	MAX_ANISO	16
#endif

#ifdef _XBOX
	#define _DEPTH_FROM_ZBUFFER_
#endif


#define	VEL_MULTIPLIER	0.5f	// can shorten blur by making this <1.0f

#define	MAX_VEL			0.03f	// max velocity (does nothing when ANISO_TRICK is defined)



texture sceneTex : TEXTURE;
sampler sceneInputTex : SAMPLER = sampler_state
{
	Texture = < sceneTex >;
	AddressU  = Clamp;
	AddressV  = Clamp;
#ifdef ANISO_TRICK
#ifdef _PS3_
	MinFilter = Linear;
	MagFilter = Linear;
	PS3MaxAnisotropy=Aniso16;
#else
	MinFilter = Anisotropic;
	MagFilter = Anisotropic;
	SET_MAX_ANISOTROPY( MAX_ANISO )
#endif
#else
	MinFilter = Linear;
	MagFilter = Linear;
	SET_NO_ANISOTROPY
#endif
#ifdef _PS3_
	FX_SAMPLERSTATE_SRGB_TEXTURE
#else
	FX_SAMPLERSTATE_LINEAR_TEXTURE
#endif
	MipFilter = None;
};

texture depthTex : TEXTURE;
sampler depthInputTex : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < depthTex >;
	AddressU  = Clamp;
	AddressV  = Clamp;
#ifdef _PS3_
	MinFilter = Point;
	MagFilter = Point;
#else
	MinFilter = Linear;
	MagFilter = Linear;
#endif	
	MipFilter = None;
	SET_NO_ANISOTROPY
};

float2 viewportOrigin;
float2 viewportScale;


struct VSINPUT
{
#ifdef _PS3_
	float3 position : POSITION;
#else	
	float3 position : POSITION;
	float2 texCoord : TEXCOORD0;
#endif	
};

struct VSOUTPUT
{
	float4 position : POSITION;
	float4 texCoord : TEXCOORD0;	// xy=texcoord, zw=pixel position
};


VSOUTPUT MotionBlurVS_WithViewport( VSINPUT _input )
{
	VSOUTPUT output;

	// Apply the viewport transformation to the input tex coord
	output.position = float4( _input.position.xyz, 1.0f );
#ifdef _PS3_
	// Generates its own coords
	float2 texCoord;
	texCoord.x=_input.position.x*0.5f+0.5f;
	texCoord.y=_input.position.y*-0.5f+0.5f;
	output.texCoord.xy = ( texCoord * viewportScale ) + viewportOrigin;
	output.texCoord.z = (texCoord.x*2.0f)-1.0f;
	output.texCoord.w = ((1.0f-texCoord.y)*2.0f)-1.0f;
#else
	output.texCoord.xy = ( _input.texCoord * viewportScale ) + viewportOrigin;
	output.texCoord.z = (_input.texCoord.x*2.0f)-1.0f;
	output.texCoord.w = ((1.0f-_input.texCoord.y)*2.0f)-1.0f;
#endif	

	return output;
}

float4x4	viewProjI;
float4x4	prevViewProj;

float	CalcDepth( float4 depthTex )
{
#ifdef _PS3_
	// calc depth using z buffer
	const float3 depthFactor = float3(65536.0f / 16777215.0f, 256.0f / 16777215.0f, 1.0f / 16777215.0f);
	float depth = dot(round(float3(depthTex.a, depthTex.r, depthTex.g) * 255.0f), depthFactor);	
	
	// Need to rescale due to GL matrix
	depth*=2.0f;
	depth-=1.0f;
#else
	// calc depth
	#ifdef _XBOX
		float	depth = depthTex.x;
	#else
		#ifdef _DEPTH_FROM_ZBUFFER_
			float	depth = (depthTex.x*255.0f/256.0f) + (depthTex.y*255.0f/65536.0f) + (depthTex.z*255.0f/16777216.0f);
		#else
			float	depth = depthTex.x;
		#endif
	#endif
#endif	

	return depth;
}

COLOUR_OUTPUT_TYPE	MotionBlurPS( float4 texCoord : TEXCOORD0 ) : COLOR0
{
	float4	output;
	
	// get textures
	const float2	uv = texCoord.xy;
	float4	depthTex = tex2D( depthInputTex, uv );

	// get this pixel's world position (in view-proj space)
	float 	d = CalcDepth( depthTex );
#ifndef _DEPTH_FROM_ZBUFFER_
	if( d==0.0f )
	{
		d = 1.0f;
	}
#endif
	float4	currentPos = float4( texCoord.z, texCoord.w, d, 1.0f );
	float4	viewPos = mul( currentPos, viewProjI );
	float4	worldPos = viewPos / viewPos.w;

	// find where this pixel was last frame
	float4	prevPos = mul( worldPos, prevViewProj );
	prevPos	/= prevPos.w;

	// calc velocity from pixel to pixel last frame
	float2	velocity = (currentPos - prevPos) * 0.5f * VEL_MULTIPLIER;
	velocity.x *= -1.0f;

#if !defined(LOW_QUALITY) && !defined(ANISO_TRICK)
	// slow down velocity at edges to avoid problems when trying to project off the screen
	float	edge = min(min(prevPos.y,prevPos.x),-prevPos.x);	// miss out -prevPos.y cos we don't worry too much about the top of the screen having this problem
	float	mul = saturate(((edge+1.0f)*2.0f)+1.0f);
	mul = (mul*0.9f)+0.1f;
	velocity *= mul;
#endif

#if !defined(ANISO_TRICK)
	// limit velocity
	float	vel_len_sq = dot(velocity,velocity);
	if( vel_len_sq>MAX_VEL*MAX_VEL )
	{
		velocity /= sqrt( vel_len_sq );
		velocity *= MAX_VEL;
	}
#endif

	// do blur
	float2	velocity_step = velocity/DIVVEL;

#ifdef ANISO_TRICK
#ifdef _PS3_
#ifdef _POST_PARSE_
	float4	col = tex2D( sceneInputTex, uv, velocity, float2(0.0f,0.0f) );
#else
	float4	col = tex2D( sceneInputTex, uv );
#endif
#else
	float4	col = tex2D( sceneInputTex, uv, velocity, float2(0.0f,0.0f) );
#endif
#else
	float2	uvsam = uv;
	float4	col = tex2D( sceneInputTex, uvsam );
	for( int i=1; i<SAMPLES; i++ )
	{
		uvsam += velocity_step;
		col += tex2D( sceneInputTex, uvsam );
	}
	col /= SAMPLES;
#endif

	// output
	output = col;

//output.rgb = tex2D( depthInputTex, uv );
//output.rgb = -pixelPos.y;
//output.rgb = 0; output.rgb = saturate(velocity.y)*2.0f;
//output.rgb = saturate(length(velocity/1.0));
//output.rgb *= mul;
//if( vel_len_sq>MAX_VEL*MAX_VEL ) {output.rgb = 1; }

	return output;
}


COLOUR_OUTPUT_TYPE	MotionBlurUsingMaskPS( float4 texCoord : TEXCOORD0 ) : COLOR0
{
	float4	output;
	
	// get textures
	const float2	uv = texCoord.xy;
	float4	depthTex = tex2D( depthInputTex, uv );

	// get this pixel's world position (in view-proj space)
	float 	d = CalcDepth( depthTex );
#ifndef _DEPTH_FROM_ZBUFFER_
	if( d==0.0f )
	{
		d = 1.0f;
	}
#endif
	float4	currentPos = float4( texCoord.z, texCoord.w, d, 1.0f );
	float4	viewPos = mul( currentPos, viewProjI );
	float4	worldPos = viewPos / viewPos.w;

	// find where this pixel was last frame
	float4	prevPos = mul( worldPos, prevViewProj );
	prevPos	/= prevPos.w;

	// calc velocity from pixel to pixel last frame
	float2	velocity = (currentPos - prevPos) * 0.5f * VEL_MULTIPLIER;
	velocity.x *= -1.0f;

#if !defined(LOW_QUALITY) && !defined(ANISO_TRICK)
	// slow down velocity at edges to avoid problems when trying to project off the screen
	float	edge = min(min(prevPos.y,prevPos.x),-prevPos.x);	// miss out -prevPos.y cos we don't worry too much about the top of the screen having this problem
	float	mul = saturate(((edge+1.0f)*2.0f)+1.0f);
	mul = (mul*0.9f)+0.1f;
	velocity *= mul;
#endif

#if !defined(ANISO_TRICK)
	// limit velocity
	float	vel_len_sq = dot(velocity,velocity);
	if( vel_len_sq>MAX_VEL*MAX_VEL )
	{
		velocity /= sqrt( vel_len_sq );
		velocity *= MAX_VEL;
	}
#endif

	// do blur
	float2	velocity_step = velocity/DIVVEL;

#ifdef ANISO_TRICK
	#ifdef _PS3_
		#ifdef _POST_PARSE_
			float4	col = tex2D( sceneInputTex, uv, velocity, float2(0.0f,0.0f) );
		#else
			float4	col = tex2D( sceneInputTex, uv );
		#endif
	#else
		float4	col = tex2D( sceneInputTex, uv, velocity, float2(0.0f,0.0f) );
	#endif
#else
	float2	uvsam = uv;
	float4	col = tex2D( sceneInputTex, uvsam );
	if( col.a>0.0f )
	{
		float		sum = 1.0f;
		for( int i=1; i<SAMPLES; i++ )
		{
			uvsam += velocity_step;
			float4	col2 = tex2D( sceneInputTex, uvsam );
			col += col2*col2.a;
			sum += col2.a;
		}
		col /= sum;
	}
#endif

	// output
	output = col;

//output.rgb = tex2D( depthInputTex, uv );
//output.rgb = -pixelPos.y;
//output.rgb = 0; output.rgb = saturate(velocity.y)*2.0f;
//output.rgb = saturate(length(velocity/1.0));
//output.rgb *= mul;
//if( vel_len_sq>MAX_VEL*MAX_VEL ) {output.rgb = 1; }

	return output;
}



struct VSOUTPUT_MASK
{
	float4 position : POSITION;
};

VSOUTPUT_MASK MaskVS( VSINPUT _input )
{
	VSOUTPUT_MASK output;

	output.position = float4( _input.position.xyz, 1.0f );

	return output;
}

COLOUR_OUTPUT_TYPE	MaskClearPS() : COLOR0
{
	float4	output;
	output = 1.0f;
	return output;
}

COLOUR_OUTPUT_TYPE	MaskPS() : COLOR0
{
	float4	output;
	output = 0.0f;
	return output;
}

COLOUR_OUTPUT_TYPE	MaskFromStencilPS( float4 uv : TEXCOORD0 ) : COLOR0
{
	float4	output;
	float4	depthTex = tex2D( depthInputTex, uv.xy );
	output = 1.0f-(depthTex.b*255.0f);	// stencil value
	return output;
}





technique MotionBlur
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx MotionBlurVS_WithViewport();
		PixelShader = compile sce_fp_rsx MotionBlurPS();
#else		
		VertexShader = compile vs_3_0 MotionBlurVS_WithViewport();
		PixelShader = compile ps_3_0 MotionBlurPS();
#endif
	}
}

technique MotionBlurUsingMask
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx MotionBlurVS_WithViewport();
		PixelShader = compile sce_fp_rsx MotionBlurUsingMaskPS();
#else		
		VertexShader = compile vs_3_0 MotionBlurVS_WithViewport();
		PixelShader = compile ps_3_0 MotionBlurUsingMaskPS();
#endif
	}
}


technique ClearMaskBeforeStencil
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx MaskVS();
		PixelShader = compile sce_fp_rsx MaskClearPS();
#else		
		VertexShader = compile vs_3_0 MaskVS();
		PixelShader = compile ps_3_0 MaskClearPS();
#endif
	}
}


technique CreateMaskFromStencil
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx MaskVS();
		PixelShader = compile sce_fp_rsx MaskPS();
#else		
		VertexShader = compile vs_3_0 MaskVS();
		PixelShader = compile ps_3_0 MaskPS();
#endif
	}
}

technique WriteMaskFromDepthStencil
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx MotionBlurVS_WithViewport();
		PixelShader = compile sce_fp_rsx MaskFromStencilPS();
#else		
		VertexShader = compile vs_3_0 MotionBlurVS_WithViewport();
		PixelShader = compile ps_3_0 MaskFromStencilPS();
#endif
	}
}
